// VoRS -- Vo(IP) Really Simple
// Copyright (C) 2024 Sergey Matveev <stargrave@stargrave.org>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as
// published by the Free Software Foundation, version 3 of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

package main

import (
	"fmt"
	"math"
	"strings"
	"sync/atomic"
	"time"

	"github.com/dustin/go-humanize"
	"github.com/jroimartin/gocui"
	vors "go.stargrave.org/vors/internal"
)

type Stats struct {
	pkts    int64
	bytes   uint64
	bads    int64
	lost    int64
	reorder int64
	last    time.Time
	vol     uint64
	volN    uint64
	dead    chan struct{}
}

func (stats *Stats) AddRMS(pcm []int16) {
	var vol uint64
	for _, s := range pcm {
		vol += uint64(int64(s) * int64(s))
	}
	atomic.AddUint64(&stats.vol, vol)
	atomic.AddUint64(&stats.volN, uint64(len(pcm)))
}

func statsDrawer(stats *Stats, name string) {
	var err error
	tick := time.Tick(vors.ScreenRefresh)
	var now time.Time
	var v *gocui.View
	var vol, volN float64
	var maxRMS float64
	var rep int
	for {
		select {
		case <-stats.dead:
			GUI.DeleteView(name)
			GUI.DeleteView(name + "-vol")
			return
		case now = <-tick:
			s := fmt.Sprintf(
				"%s  |  %s  |  B/L/R: %s/%s/%s",
				humanize.Comma(stats.pkts),
				humanize.IBytes(stats.bytes),
				humanize.Comma(stats.bads),
				humanize.Comma(stats.lost),
				humanize.Comma(stats.reorder),
			)
			if name == *Name && Muted {
				s += "  |  " + vors.CRed + "MUTE" + vors.CReset
			} else {
				if stats.last.Add(vors.ScreenRefresh).After(now) {
					s += "  |  " + vors.CGreen + "TALK" + vors.CReset
				}
			}
			v, err = GUI.View(name)
			if err == nil {
				v.Clear()
				v.Write([]byte(s))
			}
			vol = float64(atomic.SwapUint64(&stats.vol, 0))
			volN = float64(atomic.SwapUint64(&stats.volN, 0))
			v, err = GUI.View(name + "-vol")
			if err == nil {
				v.Clear()
				if volN == 0 {
					continue
				}
				vol = math.Sqrt(vol / volN)
				if vol/4 > maxRMS {
					maxRMS = vol / 4
				}
				rep = int(float64(GUIMaxY) * vol / maxRMS)
				v.Write([]byte(strings.Repeat("▒", rep)))
			}
		}
	}
}
