package main

import (
	"hash"
	"log/slog"
	"net"
	"sync"
	"time"

	"github.com/flynn/noise"
	vors "go.stargrave.org/vors/internal"
)

var (
	Peers  = map[byte]*Peer{}
	PeersM sync.Mutex
)

type Stats struct {
	pktsRx  int64
	pktsTx  int64
	bytesRx uint64
	bytesTx uint64
	bads    int64
	last    time.Time
}

type Peer struct {
	name  string
	sid   byte
	addr  *net.UDPAddr
	key   []byte
	mac   hash.Hash
	stats *Stats
	room  *Room

	logger     *slog.Logger
	conn       net.Conn
	rx, tx     chan []byte
	rxCS, txCS *noise.CipherState
	alive      chan struct{}
	aliveOnce  sync.Once
}

func (peer *Peer) Close() {
	peer.aliveOnce.Do(func() {
		close(peer.rx)
		close(peer.tx)
		close(peer.alive)
		peer.conn.Close()
	})
}

func (peer *Peer) Rx() {
	for {
		buf, err := vors.PktRead(peer.conn)
		if err != nil {
			peer.logger.Error("rx", "err", err)
			break
		}
		buf, err = peer.rxCS.Decrypt(buf[:0], nil, buf)
		if err != nil {
			peer.logger.Error("rx decrypt", "err", err)
			break
		}
		peer.rx <- buf
	}
	peer.Close()
}

func (peer *Peer) Tx() {
	for buf := range peer.tx {
		if peer.txCS == nil {
			continue
		}
		buf, err := peer.txCS.Encrypt(buf[:0], nil, buf)
		if err != nil {
			peer.logger.Error("tx encrypt", "err", err)
			break
		}
		err = vors.PktWrite(peer.conn, buf)
		if err != nil {
			peer.logger.Error("tx", "err", err)
			break
		}
	}
	peer.Close()
}
