package main

import (
	"crypto/cipher"
	"hash"
	"log/slog"
	"net"
	"sync"
	"time"

	vors "go.stargrave.org/vors/v6/internal"
)

var (
	Peers  = map[byte]*Peer{}
	PeersM sync.Mutex
)

type Stats struct {
	last    time.Time
	pktsRx  int64
	pktsTx  int64
	bads    int64
	bytesRx uint64
	bytesTx uint64
}

type Peer struct {
	name  string
	addr  *net.UDPAddr
	stats *Stats
	room  *Room
	key   []byte
	mac   hash.Hash
	muted bool
	sid   byte

	logger    *slog.Logger
	conn      *vors.NSConn
	rxAEAD    cipher.AEAD
	txAEAD    cipher.AEAD
	rxNonce   []byte
	txNonce   []byte
	rx, tx    chan []byte
	alive     chan struct{}
	aliveOnce sync.Once
}

func (peer *Peer) Close() {
	peer.aliveOnce.Do(func() {
		close(peer.rx)
		close(peer.tx)
		close(peer.alive)
		peer.conn.Conn.Close()
	})
}

func (peer *Peer) Rx() {
	var err error
	for buf := range peer.conn.Rx {
		buf, err = peer.rxAEAD.Open(buf[:0], peer.rxNonce, buf, nil)
		if err != nil {
			peer.logger.Error("rx decrypt", "err", err)
			break
		}
		peer.rx <- buf
		vors.Incr(peer.rxNonce)
	}
	peer.Close()
}

func (peer *Peer) Tx() {
	var err error
	for buf := range peer.tx {
		buf = peer.txAEAD.Seal(nil, peer.txNonce, buf, nil)
		if err = peer.conn.Tx(buf); err != nil {
			peer.logger.Error("tx write", "err", err)
			break
		}
		vors.Incr(peer.txNonce)
	}
	peer.Close()
}
