package main

import (
	"fmt"
	"sort"
	"strings"
	"sync"
	"time"

	"github.com/dustin/go-humanize"
	vors "go.stargrave.org/vors/v6/internal"
)

var (
	Rooms  = map[string]*Room{}
	RoomsM sync.RWMutex
)

type Room struct {
	alive  chan struct{}
	peers  map[byte]*Peer
	name   string
	key    string
	peersM sync.RWMutex
}

func (room *Room) Stats(now time.Time) []string {
	sids := make([]int, 0, len(room.peers))
	room.peersM.RLock()
	for sid := range room.peers {
		sids = append(sids, int(sid))
	}
	room.peersM.RUnlock()
	sort.Ints(sids)
	lines := make([]string, 0, len(sids))
	for _, sid := range sids {
		peer := room.peers[byte(sid)]
		if peer == nil {
			continue
		}
		peerFlags := []string{" ", " "}
		line := fmt.Sprintf(
			"%10s  |  %s | Rx/Tx/Bad: %s / %s / %s |  %s / %s",
			peer.name,
			peer.addr,
			humanize.Comma(peer.stats.pktsRx),
			humanize.Comma(peer.stats.pktsTx),
			humanize.Comma(peer.stats.bads),
			humanize.IBytes(peer.stats.bytesRx),
			humanize.IBytes(peer.stats.bytesTx),
		)
		if peer.muted {
			peerFlags[0] = vors.CRed + "M" + vors.CReset
		}
		if peer.stats.last.Add(vors.ScreenRefresh).After(now) {
			peerFlags[1] = vors.CGreen + "T" + vors.CReset
		}
		lines = append(lines, strings.Join(peerFlags, "")+line)
	}
	return lines
}
