package pqhs

import (
	"crypto/ecdh"
	"crypto/rand"
	"crypto/sha3"

	vors "go.stargrave.org/vors/v6/internal"
	"go.stargrave.org/vors/v6/pqhs/mceliece6960119"
	sntrup761kem "go.stargrave.org/vors/v6/pqhs/sntrup761/kem"
	sntrup761 "go.stargrave.org/vors/v6/pqhs/sntrup761/kem/ntruprime/sntrup761"
	"golang.org/x/crypto/chacha20poly1305"
)

type Client struct {
	ephPrvX25519 *ecdh.PrivateKey
	SymmetricState
}

func NewClient(
	serverStaticPubMcElieceRaw, serverStaticPubX25519Raw []byte,
) (c *Client, payload []byte, err error) {
	c = &Client{}
	var serverStaticPubMcEliece *mceliece6960119.PublicKey
	serverStaticPubMcEliece, err = mceliece6960119.UnmarshalBinaryPublicKey(
		serverStaticPubMcElieceRaw)
	if err != nil {
		return
	}
	var serverStaticPubX25519 *ecdh.PublicKey
	{
		x25519 := ecdh.X25519()
		serverStaticPubX25519, err = x25519.NewPublicKey(serverStaticPubX25519Raw)
		if err != nil {
			return
		}
		c.ephPrvX25519, err = x25519.GenerateKey(rand.Reader)
		if err != nil {
			return
		}
	}
	var ctMcEliece []byte
	var k []byte
	ctMcEliece, k, err = mceliece6960119.Encapsulate(serverStaticPubMcEliece)
	if err != nil {
		return
	}
	c.H([]byte(vors.Magic))
	c.H(sha3.SumSHAKE256(
		append(serverStaticPubMcElieceRaw, serverStaticPubX25519Raw...), 64))
	c.H(ctMcEliece)
	c.CK(k)
	payload = append(ctMcEliece,
		c.Seal(CtxClientX25519, c.ephPrvX25519.PublicKey().Bytes())...)
	k, err = c.ephPrvX25519.ECDH(serverStaticPubX25519)
	if err == nil {
		c.CK(k)
	}
	return
}

func (c *Client) Read(reply, prefinish []byte) (payload []byte, err error) {
	ctX25519 := reply[:32+chacha20poly1305.Overhead]
	ctSNTRUP := reply[len(ctX25519):]
	var k []byte
	{
		var serverEphPubX25519Raw []byte
		serverEphPubX25519Raw, err = c.Open(CtxServerX25519, ctX25519)
		if err != nil {
			return
		}
		_, k, err = DH(c.ephPrvX25519, serverEphPubX25519Raw)
		if err != nil {
			return
		}
	}
	c.CK(k)
	{
		var serverEphPubSNTRUPRaw []byte
		serverEphPubSNTRUPRaw, err = c.Open(CtxServerSNTRUP761, ctSNTRUP)
		if err != nil {
			return
		}
		sntrup761s := sntrup761.Scheme()
		var serverEphPubSNTRUP sntrup761kem.PublicKey
		serverEphPubSNTRUP, err = sntrup761s.UnmarshalBinaryPublicKey(
			serverEphPubSNTRUPRaw)
		if err != nil {
			return
		}
		ctSNTRUP, k, err = sntrup761s.Encapsulate(serverEphPubSNTRUP)
		if err != nil {
			return
		}
	}
	ctSNTRUP = c.Seal(CtxClientSNTRUP761, ctSNTRUP)
	c.CK(k)
	payload = append(ctSNTRUP, c.Seal(CtxClientPrefinish, prefinish)...)
	return
}
